/*****************************************************************************
 $Id: HistoPanel.java,v 1.4 2002/04/26 22:43:32 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.GUI;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;

import trex.*;
import java.awt.event.*;


/**
 * This class provides a panel for (possibly two concurrent) histograms
 * of ImageIcons.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.4 $
 */
public class HistoPanel extends JPanel {

  /** Display Histogram for value */
  public static final int CHANNEL_VALUE = 0;
  /** Display Histogram for red channel */
  public static final int CHANNEL_RED   = 1;
  /** Display Histogram for green channel */
  public static final int CHANNEL_GREEN = 2;
  /** Display Histogram for blue channel */
  public static final int CHANNEL_BLUE  = 3;

  Frame parent;

  /** The ImageIcons to draw histograms for */
  ImageIcon img1;
  ImageIcon img2;

  /** Colors for the two histograms */
  Color color1;
  Color color2;

  /** number of bins */
  final static int bin_count = 256;
  /** Width of histoggram display is broaden x bin_count */
  final static int broaden = 2;
  final static int width = bin_count*broaden;
  int height;

  /** The channel to draw with this instance */
  int channel;

  /** Label explaining the colors (color1/2) */
  String colExpl;

  /** Windows' tool bar */
  JToolBar jToolBarHistoPanel = new JToolBar();
  /** Tool bar radio buttons */
  ButtonGroup buttonGroupChannelSelect = new ButtonGroup();
  JRadioButton jSelectValue = new JRadioButton();
  JRadioButton jSelectRed = new JRadioButton();
  JRadioButton jSelectBlue = new JRadioButton();
  JRadioButton jSelectGreen = new JRadioButton();
  /** Windows' layout */
  BorderLayout borderLayoutHistoPanel = new BorderLayout();
  /** Put stuff into a JScrollPane */
  JScrollPane jScrollPaneHisto = new JScrollPane();
  /** Text label for the explanation string (colExpl) */
  JLabel jLabelColExpl = new JLabel();
  /** Image label for the histogram itself */
  JLabel jLabelHisto = new JLabel();
  /** Panel to put labels in */
  JPanel jPanelInner = new JPanel();
  /** Inner panel's Layout */
  BorderLayout borderLayoutInner = new BorderLayout();

  /**
   * Create histogram bins.
   * All pixels from the array are counted and put in to their respective bins.
   * @param pixels Pixel data.
   * @param channel Channel to display (CHANNEL_VALUE, CHANNEL_RED, CHANNEL_GREEN or CHANNEL_BLUE).
   * @return The bin array.
   */
  protected static int[] collectBins(int pixels[], int channel) {
    /* create bin array */
    int bins[] = new int[bin_count];
    int val = 0;
    Color col;

    for (int i = 0; i < bins.length; i++)
      bins[i] = 0;

    for (int i = 0; i < pixels.length; i++) {
      col = new Color(pixels[i]);
      switch (channel) {
        case CHANNEL_VALUE:  // for "value": get brightness of all 3 channels, sum up and divide by 3.
          val = (int)(col.getRed() + col.getGreen() + col.getBlue())/3;
          break;
        case CHANNEL_RED:
          val = col.getRed(); break;
        case CHANNEL_GREEN:
          val = col.getGreen(); break;
        case CHANNEL_BLUE:
          val = col.getBlue(); break;
      }

      bins[val]++;
    }
    return bins;
  }

  /**
   * Grab pixels from image and create histogram bins.
   * In fact this method only grabs pixel data and alls {@link #collectBins(int[], int)}.
   * @param parent Parent window.
   * @param img Image to grab pixels from.
   * @param channel Channel to create bins of (see static fields).
   * @return The bin array.
   */
  protected static int[] collectBins(Frame parent, ImageIcon img, int channel) {
    int w = img.getIconWidth();
    int h = img.getIconHeight();

    int[] pixels = new int[h*w];

    PixelGrabber pg = new PixelGrabber(img.getImage(), 0, 0, w, h, pixels, 0, w);

    try {
      pg.grabPixels();
    }

    catch (Exception e) {
      JOptionPane.showMessageDialog(parent,
                      "An unknown error occured transforming image data. Sorry.",
                      "Error converting data",
                      JOptionPane.ERROR_MESSAGE);
      return null;
    }

    return collectBins(pixels, channel);
  }

  /**
   * Draw a single line in the image according to the supplied histo bin.
   * @param bi Image to draw to.
   * @param border Empty (unused) border around histogram.
   * @param x Horizontal position of line.
   * @param max Maximum value contained in histo bins.
   * @param val Relative height of this histo line.
   * @param height Image height.
   */
  private static void drawHistoLine(BufferedImage bi,
                                    int border,
                                    int x,
                                    int max,
                                    int val,
                                    int height,
                                    Color col) {
    double share = ((double)val)/((double)max);
    double paintable = (double)(height-(2*border));

    double dlen = share * paintable;
    int len = (int)dlen;

    for (int i = height-border; i > (height-border)-len; i--) {
      bi.setRGB(x+border, i, (col.getRGB() | 0xFF000000));
    }
  }

  /**
   * Create the actual histogram image.
   * @param parent Parent window (for Message boxes).
   * @param img Image to create histogram for.
   * @param border Empty (unused) border around actual histogram.
   * @param height Image height.
   * @param color Color of histo lines.
   * @param channel The channel to draw histogram for (CHANNEL_VALUE, CHANNEL_RED, CHANNEL_GREEN or CHANNEL_BLUE).
   * @return An ImageIcon containing the histogram.
   */
  public static ImageIcon createHisto(Frame parent,
                                      ImageIcon img,
                                      int border,
                                      int height,
                                      Color color,
                                      int channel) {
    BufferedImage result = new BufferedImage(width+(2*border), height, BufferedImage.TYPE_INT_ARGB);

    /* draw rectangle around histogram */
    for (int i = border; i < height-border; i++) {   // top-down
      result.setRGB(border, i, 0xFF000000);
      result.setRGB(border+width, i, 0xFF000000);
    }
    for (int i = border; i < width+border; i++) {      // left-right
      result.setRGB(i, border, 0xFF000000);
      result.setRGB(i, height-border, 0xFF000000);
    }

    int bins[] = collectBins(parent, img, channel);

    int max = TRexUtil.maxArray(bins);
    for (int i = 0; i < bin_count; i++) {
      drawHistoLine(result, border+1, broaden*i, max, bins[i], height, color);
    }

    return new ImageIcon(result);
  }


  /**
   * This calls the static method
   * {@link #createHisto(Frame, ImageIcon, int, int, Color, int)},
   * setting default values.
   * @param img Image to draw histogram for.
   * @param color Color of histo lines.
   * @return ImageIcon containing the histogram.
   */
  ImageIcon createHisto(ImageIcon img, Color color) {
    return createHisto(parent, img, 10, height, color, channel);
  }


  /**
   * Recalculates the histograms and resets the histo label.
   */
  void resetHistos() {
    /*
     * Depending on set images, set no, one or a composited ImageIcon
     */
    if ((img1 != null) && (img2 != null)) {
      BufferedImage added = TRexUtil.simpleAddImages(parent,
                                                     createHisto(img1, color1),
                                                     createHisto(img2, color2));
      jLabelHisto.setIcon(new ImageIcon(added));
      jLabelColExpl.setText(colExpl);
    } else if ((img1 != null) && (img2 == null)) {
      jLabelHisto.setIcon(createHisto(img1, color1));
      jLabelColExpl.setText(colExpl);
    } else if ((img1 == null) && (img2 != null)) {
      jLabelHisto.setIcon(createHisto(img2, color2));
      jLabelColExpl.setText(colExpl);
    } else if ((img1 == null) && (img2 == null)) {
      jLabelHisto.setIcon(null);
      jLabelColExpl.setText("Histogram. No images available.");
    }
  }

  /**
   * Set image1.
   * @param img Image to set as image 1.
   */
  public void setImg1(ImageIcon img) {
    this.img1 = img;
    resetHistos();
  }

  /**
   * Set image2.
   * @param img Image to set as image 2.
   */
  public void setImg2(ImageIcon img) {
    this.img2 = img;
    resetHistos();
  }

  /**
   * Constructor. Initializes Data and Panel.
   * @param parent Parent window (for message boxes).
   * @param img1 First image to draw histogram for.
   * @param img2 Second image to draw histogram for.
   * @param color1 Color of first histogram.
   * @param color2 Color of second histogram.
   * @param colExpl Text string to inform user of color uses.
   * @param height Image height.
   * @param channel The channel to draw histogram for (CHANNEL_VALUE, CHANNEL_RED, CHANNEL_GREEN or CHANNEL_BLUE).
   */
  public HistoPanel(Frame parent,
                    ImageIcon img1,
                    ImageIcon img2,
                    Color color1,
                    Color color2,
                    String colExpl,
                    int height,
                    int channel) {

    // store data
    this.parent = parent;

    this.img1 = img1;
    this.img2 = img2;

    this.color1 = color1;
    this.color2 = color2;

    this.height = height;

    this.channel = channel;

    this.colExpl = colExpl;

    try {
      jbInit();
    }
    catch(Exception e) {
      e.printStackTrace();
    }

    switch (channel) {
      case CHANNEL_VALUE: jSelectValue.setSelected(true); break;
      case CHANNEL_RED:   jSelectRed.setSelected(true); break;
      case CHANNEL_GREEN: jSelectGreen.setSelected(true); break;
      case CHANNEL_BLUE:  jSelectBlue.setSelected(true); break;
    }

    resetHistos();
  }

  /** Init the window. JBuilder wizardry. :) */
  private void jbInit() throws Exception {
    this.setLayout(borderLayoutHistoPanel);
    jSelectValue.setText("Value");
    jSelectValue.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jSelectValue_actionPerformed(e);
      }
    });
    jSelectRed.setText("Red");
    jSelectRed.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jSelectRed_actionPerformed(e);
      }
    });
    jSelectGreen.setText("Green");
    jSelectGreen.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jSelectGreen_actionPerformed(e);
      }
    });
    jSelectBlue.setText("Blue");
    jSelectBlue.addActionListener(new java.awt.event.ActionListener() {
      public void actionPerformed(ActionEvent e) {
        jSelectBlue_actionPerformed(e);
      }
    });
    jPanelInner.setLayout(borderLayoutInner);
    this.add(jToolBarHistoPanel, BorderLayout.NORTH);
    jToolBarHistoPanel.add(jSelectValue, null);
    jToolBarHistoPanel.add(jSelectRed, null);
    jToolBarHistoPanel.add(jSelectGreen, null);
    jToolBarHistoPanel.add(jSelectBlue, null);
    this.add(jScrollPaneHisto, BorderLayout.CENTER);
    buttonGroupChannelSelect.add(jSelectValue);
    buttonGroupChannelSelect.add(jSelectRed);
    buttonGroupChannelSelect.add(jSelectBlue);
    buttonGroupChannelSelect.add(jSelectGreen);
    jPanelInner.add(jLabelHisto, BorderLayout.CENTER);
    jPanelInner.add(jLabelColExpl, BorderLayout.SOUTH);
    jScrollPaneHisto.getViewport().add(jPanelInner);
  }

  /** Channel switching event handler. */
  void jSelectValue_actionPerformed(ActionEvent e) {
    channel = CHANNEL_VALUE;
    resetHistos();
  }

  /** Channel switching event handler. */
  void jSelectRed_actionPerformed(ActionEvent e) {
    channel = CHANNEL_RED;
    resetHistos();
  }

  /** Channel switching event handler. */
  void jSelectGreen_actionPerformed(ActionEvent e) {
    channel = CHANNEL_GREEN;
    resetHistos();
  }

  /** Channel switching event handler. */
  void jSelectBlue_actionPerformed(ActionEvent e) {
    channel = CHANNEL_BLUE;
    resetHistos();
  }
}