/*****************************************************************************
 $Id: TRexUtil.java,v 1.17 2002/04/28 21:37:26 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;

import trex.GUI.*;

/**
 * This class provides a couple of helper functions commonly used in TRex
 * (and maybe others... ;)
 * Methods are static and can be called from outside without instantiating
 * TRexUtil.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.17 $
 */

public class TRexUtil {

/* ************************************************ */
/*                 Panel creation                   */
/* ************************************************ */

  /**
   * Creates a JPanel only containing a text label.
   * @param text Text to display in panel's label.
   * @return Panel containing text.
   */
  public static JPanel makeTextPanel(String text) {
    JPanel panel = new JPanel(false);
    JLabel filler = new JLabel(text);
    filler.setHorizontalAlignment(JLabel.CENTER);
    panel.setLayout(new GridLayout(1, 1));
    panel.add(filler);
    return panel;
  }

  /**
   * Creates a JPanel only containing an {@link javax.swing.ImageIcon}.
   * @param img Icon to display.
   * @return Panel containing an ImageIcon.
   */
  public static Component makeImagePanel(ImageIcon img) {
    JPanel panel = new JPanel(false);
    JLabel filler = new JLabel(img);
    filler.setHorizontalAlignment(JLabel.CENTER);
    panel.setLayout(new GridLayout(1, 1));
    panel.add(filler);
    return panel;
  }

  /**
   * Creates a JPanel only containing an {@link javax.swing.ImageIcon}
   * read from a file.
   * @param filename File containing the image.
   * @return Panel containin an ImageIcon.
   */
  public static Component makeImagePanel(String filename) {
    return makeImagePanel(new ImageIcon(filename));
  }

  /**
   * Construct a panel displaying an ImageIcon (if available) or a
   * text (as a fallback).
   * @param img Image to display, or null
   * @param text Text to display if img is null
   * @return Panel containing either an ImageIcon or a text
   */
  public static Component makePanel(ImageIcon img, String text) {
    if (img == null) return makeTextPanel(text);
    else return makeImagePanel(img);
  }

  /**
   * Construct a panel displaying an ImageIcon (if available) or a
   * text (as a fallback).
   * @param filename File containing image to display, or null
   * @param text Text to display if filename is null
   * @return Panel containing either an ImageIcon or a text
   */
  public static Component makePanel(String filename, String text) {
    if (filename == null) return makeTextPanel(text);
    else return makeImagePanel(filename);
  }

/* ************************************************ */
/*      Bit shifting wizardry                       */
/* ************************************************ */


  /**
   * Return a bit (0 || 1) from a byte array indexed as pos.
   * @param in byte array to get bit from.
   * @param pos bit number.
   * @return A bit
   */
  public static byte getBit(byte[] in, int pos) {
    // in what byte is the bit contained?
    int concernedByte = pos/8;
    // which bit in that byte is it?
    byte concernedBit  = (byte)(pos % 8);
    // create a bitmask for that bit
    byte mask = (byte)(1 << (7-concernedBit));

    // extract bit
    byte res = (byte)(in[concernedByte] & mask);
    // if result is 0, the bit was unset (return 0), else it was set (return 1)
    if (res != 0)
      return 1;
    else
      return 0;
  }

  /**
   * Set a bit in a byte array.
   * In fact, this _could_ set more than only one bit at a time - but behaviour
   * is unknown, so set value _only_ to 0 or 1.
   * @param in byte array to set bit in
   * @param pos bit number in array
   * @param value value to set (0 || 1)
   */
  public static void setBit(byte[] in, int pos, int value) {
    int concernedByte = pos/8;
    int concernedBit  = pos % 8;
    int mask = 1 << (7-concernedBit);

    if (value == 0)
      in[concernedByte] = (byte)(in[concernedByte] & (byte)(~mask));
    else
      in[concernedByte] = (byte)(in[concernedByte] | (byte)mask);
  }

  /**
   * Convert an integer to an array of 4 bytes.
   * @param in integer to transform.
   * @return the byte array.
   */
  public static byte[] intToBytes(int in) {
    byte res[] = new byte[4];
    // bit shifting is obvious; "& 0xff" may be unnecessary, but for the sake of it...
    res[0] = (byte)((in >> 24) & 0xff);
    res[1] = (byte)((in >> 16) & 0xff);
    res[2] = (byte)((in >>  8) & 0xff);
    res[3] = (byte)((in      ) & 0xff);
    return res;
  }

  /**
   * generate an integer from a byte array with 4 elements
   * @param in Input byte array.
   * @return int value consisting of input.
   */
  public static int bytesToInt(byte[] in) {
    int temp[] = new int[4];
    temp[0] = ((in[0] << 24) & 0xff000000);
    temp[1] = ((in[1] << 16) & 0x00ff0000);
    temp[2] = ((in[2] <<  8) & 0x0000ff00);
    temp[3] = ((in[3]      ) & 0x000000ff);
    return (temp[0] | temp[1] | temp[2] | temp[3]);
  }

/* ************************************************ */
/*                   Array handling                 */
/* ************************************************ */


  /**
   * concatenate two byte arrays.
   * @param in1 First input array.
   * @param in2 Latter input array.
   * @return concatenated byte array.
   */
  public static byte[] concatByteArrays(byte[] in1, byte[] in2) {
    byte res[] = new byte[in1.length + in2.length];
    for (int i = 0; i < in1.length; i++)
      res[i] = in1[i];

    for (int i = 0; i < in2.length; i++)
      res[i+in1.length] = in2[i];

    return res;
  }

  /**
   * return a part of an array.
   * @param in Byte array.
   * @param start First element to return.
   * @param len Number of elements to return.
   * @return Byte array with desired elements.
   */
  public static byte[] subArray(byte[] in, int start, int len) {
    byte res[] = new byte[len];
    for (int i = start; i < start+len; i++)
      res[i-start] = in[i];
    return res;
  }

  /**
   * print a byte array as a comma seperated string
   * @param in Byte array.
   * @return String representation of array.
   */
  public static String bytesToString(byte[] in) {
    String res = "";
    for (int i = 0; i < in.length; i++)
      res = res.concat(in[i] + ", ");

    return res;
  }

  /**
   * print a int array as a comma seperated string
   * @param in Int array.
   * @return String representation of array.
   */
  public static String intsToString(int[] in) {
    String res = "";
    for (int i = 0; i < in.length; i++)
      res = res.concat(in[i] + ", ");

    return res;
  }

  /**
   * Find largest value in an array.
   */
  public static int maxArray(int array[]) {
    int max = Integer.MIN_VALUE;

    for (int i = 0; i < array.length; i++)
      if (array[i] > max) max = array[i];

    return max;
  }

/* ************************************************ */
/*              Image handling                      */
/* ************************************************ */

  /**
   * create a BufferedImage from an ImageIcon
   * @param ii An {@link javax.swing.ImageIcon} to transform
   * @param parent A Frame to display error messages in
   * @return Buffered image with same contents
   */
  public static BufferedImage iitobi(ImageIcon ii, Frame parent) {

    int w = ii.getIconWidth();
    int h = ii.getIconHeight();

    int[] pixels = new int[h*w];


    PixelGrabber pg = new PixelGrabber(ii.getImage(), 0, 0, w, h, pixels, 0, w);

    try {
      pg.grabPixels();
    }

    catch (Exception e) {
      JOptionPane.showMessageDialog(parent,
                      "An unknown error occured transforming image data. Sorry.",
                      "Error converting data",
                      JOptionPane.ERROR_MESSAGE);
      return null;
    }

    BufferedImage bi = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);

    bi.setRGB(0, 0, w, h, pixels, 0, w);

    return bi;
  }

  /**
   * This method adds two images by returning their "OR" picture.
   * This can be used to construct a picture consisting of two channels from
   * original pics.
   * @param parent Parent window.
   * @param img1 First image.
   * @param img2 Second image.
   * @param w Picture's width.
   * @param h Picutre's height.
   */
  public static BufferedImage simpleAddImages(Frame parent,
                                              Image img1, Image img2,
                                              int w, int h) {
    int[] pixels1 = new int[h*w];
    int[] pixels2 = new int[h*w];
    int[] pixelsAdd = new int[h*w];

    PixelGrabber pg1 = new PixelGrabber(img1, 0, 0, w, h, pixels1, 0, w);
    PixelGrabber pg2 = new PixelGrabber(img2, 0, 0, w, h, pixels2, 0, w);

    try {
      pg1.grabPixels();
      pg2.grabPixels();
    }

    catch (Exception e) {
      JOptionPane.showMessageDialog(parent,
                      "An unknown error occured transforming image data. Sorry.",
                      "Error converting data",
                      JOptionPane.ERROR_MESSAGE);
      return null;
    }

    for (int i = 0; i < pixels1.length; i++)
      pixelsAdd[i] = (pixels1[i] | pixels2[i]);

    BufferedImage bi = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);

    bi.setRGB(0, 0, w, h, pixelsAdd, 0, w);

    return bi;
  }

  /**
   * This method adds two images and returns their "OR" picture.
   * This can be used to construct a picture consisting of two channels from
   * original pics.
   * @param parent Parent window.
   * @param img1 First image.
   * @param img2 Second image.
   */
  public static BufferedImage simpleAddImages(Frame parent,
                                              ImageIcon img1, ImageIcon img2) {
    return simpleAddImages(parent,
         img1.getImage(), img2.getImage(),
         img1.getIconWidth(), img1.getIconHeight());
  }

/* ************************************************ */
/*              Other helper functions              */
/* ************************************************ */

  public static void printDebug(String x) {
    //System.err.println(x);
  }
}