/*****************************************************************************
 $Id: StegoAlgoFFT.java,v 1.4 2002/05/13 17:49:08 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.Algo;

import trex.*;

import java.awt.*;
import java.awt.image.*;
import java.awt.image.renderable.*;
import javax.swing.*;

import javax.media.jai.*;
import javax.media.jai.operator.*;
import javax.media.jai.iterator.*;


/**
 * This algorithm stores data in the frequency domain.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.4 $
 */

public class StegoAlgoFFT extends StegoAlgo {

  /**
   * To increase chances to get right data from image, put each imput data
   * byte (amount) subsequent times into the image.
   */
  int amount = 1;

  /**
   * To even more increase chances to get correct decrypt data, multiply each
   * embedded byte with (factor).
   */
  int factor = 10;

  JPanel configPanel = null;
  JCheckBox cbValid = null;

  /** config panel's radio buttons. */
  ButtonGroup buttonGroup;

  /** Don't touch type conversion */ JRadioButton radioDefault;
  /** Store FFT data as bytes */ JRadioButton radioBYTE;
  /** Store FFT data as integers */ JRadioButton radioINT;
  /** Store FFT data as short integers */ JRadioButton radioSHORT;
  /** Store FFT data as unsigned short integers */ JRadioButton radioUSHORT;
  /** Store FFT data as double */ JRadioButton radioDOUBLE;
  /** Store FFT data as float */ JRadioButton radioFLOAT;
  /** JAI's sense of "unknown" */ JRadioButton radioUNDEFINED; // UNUSED! leads to Exception

  /** Spinner for "amount" value */ JSpinner spinnerAmount;
  /** Spinner for "factor" value */ JSpinner spinnerFactor;

  /**
   * Algorithm currently does not know how big an image has to be.
   * @return true.
   */
  public boolean pictureLargeEnough(ImageIcon img, String data) {

    int height = img.getIconHeight();
    int width = img.getIconWidth();
    // start with 1x1 pic.
    int squaredWidth = 1;
    int squaredHeight = 1;

    // get minimum square numbers.
    while (squaredWidth < width) squaredWidth *= 2;
    while (squaredHeight < height) squaredHeight *= 2;

    int picsize = squaredWidth * squaredHeight;

    return (data.length()+4 <= picsize);
  }

  /**
   * Algorithm doesn't have a pass phrase - uses a config dialog instead.
   * @return false.
   */
  public boolean hasPassPhrase() { return false; }

  /**
   * Algorithm has a config dialog.
   * @return true.
   */
  public boolean hasConfigDialog() { return true; }

  /**
   * Config is valid when it's "I know it's senseless" button is checked.
   * @return config validity.
   */
  public boolean validConfig() {

    if (cbValid == null) constructConfigPanel();
    if (cbValid == null) return false;
    return cbValid.isSelected();

    //return true;
  }

  /**
   * Get actual data from image.
   * @param img (idft) Image to get data from.
   * @param w Image width
   * @param h Image height
   * @return contained string.
   */

  protected String extractData(PlanarImage img, int w, int h) throws DecryptImpossibleException {

    byte redundantBytes[] = new byte[w*h]; // more than enough.
    byte dataBytes[] = new byte[w*h]; // (amount) times more than enough.
    RookIter iter = RookIterFactory.create(
                              img.getData(), null);

    iter.endBands(); iter.endPixels(); iter.endLines();

    for (int i = 0; i < w*h; i++) {
      long res = 0;
      for (int n = 0; n < amount; n++) {
        res += iter.getSample();
      //redundantBytes[i] = (byte)iter.getSample();
        if (!advanceIter(iter)) return "ERROR DECRYPTING!";
      }
      System.err.println("Result is " + res);
      dataBytes[i] = (byte)(res / (amount*factor));
    }
/*
    for (int i = 0; i < redundantBytes.length/amount; i++) {
      long res = 0;
      for (int n = 0; n < amount; n++)
        res += redundantBytes[(i*amount) + n];
      dataBytes[i] = (byte)(res / (amount*factor));
    }
*/
    // For testing: Set an arbitrary text size.
/*
    dataBytes[0] = 0;
    dataBytes[1] = 0;
    dataBytes[2] = 0;
    dataBytes[3] = 4;
*/
    // Get output string from data.
    int datalen = TRexUtil.bytesToInt(TRexUtil.subArray(dataBytes, 0, 4));

    for (int i = 0; i < 4+4; i++)
      System.err.println(dataBytes[i]);

    if ((datalen < 0) || (datalen > dataBytes.length-4))
      throw new DecryptImpossibleException();

    return new String(TRexUtil.subArray(dataBytes, 4, datalen));
  }

  /**
   * Get embedded data from image.
   * @param img Image to get data from.
   * @return Contained data.
   */
  public String getDecrypted(ImageIcon img) throws trex.DecryptImpossibleException {
    BufferedImage src = TRexUtil.iitobi(img, null);

    /**
     * FFT needs input (and gives output) that is of size
     * n^2 x m^2 with any n and m.
     *
     * Calculate this size in squaredSize and apply
     * "BorderExtWrap" JAI operation.
     *
     * Although JAI's FFT does indeed work with pictures of other sizes,
     * it returns a "squared" picture; "pre-squaring" should increase accuracy.
     */

    int oriWidth = img.getIconWidth();
    int oriHeight = img.getIconHeight();

    PlanarImage squared = squaredImage(src, oriWidth, oriHeight);

    // the config Dialog maybe needs to be constructed before getting current
    // DataBuffer type.
    getConfigDialog();
    int type = getConfigData();

    PlanarImage dft;

    // Convert with currently selected (or default) DataBuffer type.
    if (type != -1) {
      SampleModel sm = RasterFactory.createComponentSampleModel(squared.getSampleModel(),
                                                                type,
                                                                squared.getWidth(),
                                                                squared.getHeight(),
                                                                1);
      ImageLayout il = new ImageLayout();
      il.setSampleModel(sm);
      RenderingHints rh = new RenderingHints(JAI.KEY_IMAGE_LAYOUT, il);

      ParameterBlock pb = (new ParameterBlock()).addSource(squared);
      dft = JAI.create("dft", pb, rh);
    } else {
      dft = JAI.create("dft", squared);
    }

    // Now get data from image...
    String s = extractData(dft, dft.getWidth(), dft.getHeight());

    return s;
    //return "Decryption currently impossible with this algorithm.";
  }

  /**
   * Which radio button is selected?
   * @return a DataBuffer.TYPE_... value or -1 for default.
   */
  private int getConfigData() {

    amount = ((SpinnerNumberModel)(spinnerAmount.getModel())).getNumber().intValue();
    factor = ((SpinnerNumberModel)(spinnerFactor.getModel())).getNumber().intValue();

    if (radioDefault.isSelected())
      return -1;
    else if (radioBYTE.isSelected())
      return DataBuffer.TYPE_BYTE;
    else if (radioINT.isSelected())
      return DataBuffer.TYPE_INT;
    else if (radioSHORT.isSelected())
      return DataBuffer.TYPE_SHORT;
    else if (radioUSHORT.isSelected())
      return DataBuffer.TYPE_USHORT;
    else if (radioDOUBLE.isSelected())
      return DataBuffer.TYPE_DOUBLE;
    else if (radioFLOAT.isSelected())
      return DataBuffer.TYPE_FLOAT;
    else if (radioUNDEFINED.isSelected())
      return DataBuffer.TYPE_UNDEFINED;
    else
      return -1;
  }

  /**
   * Constructs the config panel.
   */
  private void constructConfigPanel() {

    /** Info checkbox: algorithm currently not doing anything. */
    cbValid = new JCheckBox("Yes, I understand that this algorithm does not allow decrypting of data.",
                            true);

    JPanel amountPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
    JPanel factorPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));

    // construct Type radio buttons and add them to a ButtonGroup.
    radioDefault = new JRadioButton("DFT Type: Default", true);
    radioBYTE = new JRadioButton("Byte");
    radioINT = new JRadioButton("Integer");
    radioSHORT = new JRadioButton("Short");
    radioUSHORT = new JRadioButton("Unsigned Short");
    radioDOUBLE = new JRadioButton("Double");
    radioFLOAT = new JRadioButton("Float");
    radioUNDEFINED = new JRadioButton("Undefined");

    amountPanel.add(spinnerAmount = new JSpinner(new SpinnerNumberModel(amount, 1, 100, 1)));
    amountPanel.add(new JLabel("Redundancy count"));
    factorPanel.add(spinnerFactor = new JSpinner(new SpinnerNumberModel(factor, 1, 100, 1)));
    factorPanel.add(new JLabel("Multiplication factor"));

    buttonGroup = new ButtonGroup();
    buttonGroup.add(radioDefault);
    buttonGroup.add(radioBYTE);
    buttonGroup.add(radioINT);
    buttonGroup.add(radioSHORT);
    buttonGroup.add(radioUSHORT);
    buttonGroup.add(radioDOUBLE);
    buttonGroup.add(radioFLOAT);
    buttonGroup.add(radioUNDEFINED);

    /** Construct the Panel and add Components. */
    configPanel = new JPanel(new GridLayout(10,1));
    configPanel.add(cbValid);
    configPanel.add(radioDefault);
    configPanel.add(radioBYTE);
    configPanel.add(radioINT);
    configPanel.add(radioSHORT);
    configPanel.add(radioUSHORT);
    configPanel.add(radioDOUBLE);
    configPanel.add(radioFLOAT);
    //configPanel.add(radioUNDEFINED); // Undefined leads to Exception.

    configPanel.add(amountPanel);
    configPanel.add(factorPanel);

  }

  /**
   * Return the config dialog. Construct it beforehand if necessary.
   * @return An instance of the config dialog.
   */
  public Component getConfigDialog() {
    if (configPanel == null)
      constructConfigPanel();
    return configPanel;
  }

  /**
   * Static info.
   * @return "LSB in frequency domain (Fourier transformed)"
   */
  public String getInfo() {
    return "LSB in frequency domain (Fourier transformed)";
  }


  /**
   * Reformat the image to "TYPE_BYTE" so it can be displayed.
   * @param src Image to transform.
   * @param template Image to take sample- and color model from.
   * @return The transformed image.
   */
  protected PlanarImage FormatTransformed(PlanarImage src, BufferedImage template) {
    /* This was more or less stolen from sample programs. */

    // Create a RenderingHints object with desirable layout.
    ImageLayout il = new ImageLayout();
    il.setSampleModel(template.getSampleModel());
    il.setColorModel(template.getColorModel());
    RenderingHints rh = new RenderingHints(JAI.KEY_IMAGE_LAYOUT, il);

    // Convert the data to byte.
    ParameterBlock pb = new ParameterBlock();
    pb.addSource(src);
    pb.add(DataBuffer.TYPE_BYTE);
    PlanarImage dst = JAI.create("format", pb, rh);

    return dst;
  }

  private boolean advanceIter(RookIter iter) {
    if(iter.prevBandDone()) {
      iter.endBands();
      if (iter.prevPixelDone()) {
        iter.endPixels();
        if (iter.prevLineDone()) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Embed data into image. Currently a NOP.
   * @param data Data to embed.
   * @param img Envelope image.
   * @param w Image width.
   * @param h Image height.
   * @return Stego-image containing the data.
   */
  protected PlanarImage embedData(String data, PlanarImage img, int w, int h) {

    byte dataBytes[] = null;
    byte redundantBytes[] = null;

    if (data != null) {
      dataBytes = TRexUtil.concatByteArrays(
                                  TRexUtil.intToBytes(data.length()),
                                  data.getBytes());
      redundantBytes = new byte[dataBytes.length*amount];
    }

    // data from picture seems to be writable... phew!
    WritableRookIter iter = RookIterFactory.createWritable(
                              (WritableRaster)(img.getData()), null);
    iter.endBands(); iter.endPixels(); iter.endLines();

    for (int i = 0; i < dataBytes.length; i++)
      for (int n = 0; n < amount; n++)
        redundantBytes[(i*amount) + n] = dataBytes[i];

    for (int i = 0; i < redundantBytes.length; i++) {
      iter.setSample(redundantBytes[i]*factor);
      System.err.println(redundantBytes[i] + " - " + iter.getSample());
      if (!advanceIter(iter)) return null;
    }

    iter.endBands(); iter.endPixels(); iter.endLines();
    for (int i = 0; i < 40; i++) {
      System.err.println("After setting: " + iter.getSample());
      advanceIter(iter);
    }


    return img;
  }

  /**
   * As FFT needs Images of the dimension n^2 x m^2 for any natural n and m,
   * the image is converted to an image of that size beforehand with this
   * method. It utilizes JAI's BorderExtenderWrap class, as Fourier
   * transformation inherently works on periodic data.
   * After embedding and idft, the image is cropped to it's original size with
   * the {@link #cropImage} function.
   * @param img Image to resize. Any format recognized by JAI is allowed.
   * @param width Original image width.
   * @param height Original image height.
   * @return The resized image.
   */
  protected PlanarImage squaredImage(Object img, int width, int height) {
    // start with 1x1 pic.
    int squaredWidth = 1;
    int squaredHeight = 1;

    // get minimum square numbers.
    while (squaredWidth < width) squaredWidth *= 2;
    while (squaredHeight < height) squaredHeight *= 2;

    // create JAI parameter block and return created image.
    ParameterBlock pb = new ParameterBlock();
    pb.addSource(img);
    pb.add((int)0);
    pb.add((int)squaredWidth-width);
    pb.add((int)0);
    pb.add((int)squaredHeight-height);
    pb.add(BorderExtender.createInstance(BorderExtender.BORDER_WRAP));

    return JAI.create("border", pb);
  }

  /**
   * After the inverse DFT, the image can be cropped to the original image's
   * size with this function.
   * @param img Image to crop.
   * @param width New image width.
   * @param height New image height.
   * @return The cropped Image.
   */
  protected PlanarImage cropImage(Object img, int width, int height) {
    ParameterBlock pb = new ParameterBlock();
    pb.addSource(img);
    pb.add((float)0);
    pb.add((float)0);
    pb.add((float)width);
    pb.add((float)height);

    return JAI.create("crop", pb);

  }

  /**
   * This method is called from the application's StegData object. It calls
   * the JAI DFT/IDFT operations and starts the data embedding via the
   * {@link #embedData} method.
   */
  public ImageIcon getEncrypted(String data, ImageIcon img) {

    BufferedImage src = TRexUtil.iitobi(img, null);

    /**
     * FFT needs input (and gives output) that is of size
     * n^2 x m^2 with any n and m.
     *
     * Calculate this size in squaredSize and apply
     * "BorderExtWrap" JAI operation.
     *
     * Although JAI's FFT does indeed work with pictures of other sizes,
     * it returns a "squared" picture; "pre-squaring" should increase accuracy.
     */

    int oriWidth = img.getIconWidth();
    int oriHeight = img.getIconHeight();

    PlanarImage squared = squaredImage(src, oriWidth, oriHeight);

    // the config Dialog maybe needs to be constructed before getting current
    // DataBuffer type.
    getConfigDialog();
    int type = getConfigData();

    PlanarImage dft;

    // Convert with currently selected (or default) DataBuffer type.
    if (type != -1) {
      SampleModel sm = RasterFactory.createComponentSampleModel(squared.getSampleModel(),
                                                                type,
                                                                squared.getWidth(),
                                                                squared.getHeight(),
                                                                1);
      ImageLayout il = new ImageLayout();
      il.setSampleModel(sm);
      RenderingHints rh = new RenderingHints(JAI.KEY_IMAGE_LAYOUT, il);

      ParameterBlock pb = (new ParameterBlock()).addSource(squared);
      dft = JAI.create("dft", pb, rh);
    } else {
      dft = JAI.create("dft", squared);
    }

    // embed data
    PlanarImage filtered = embedData(data, dft,  img.getIconWidth(), img.getIconHeight());

    // inverse transformation
    PlanarImage idft = JAI.create("idft", filtered);

    // crop image
    PlanarImage cropped = cropImage(idft, oriWidth, oriHeight);

    // create an ImageIcon and return.
    return new ImageIcon(FormatTransformed(cropped, src).getAsBufferedImage());

  }


  /**
   * Return the default amplification of this algorithm for the combinedPanel.
   * @return Default amplification: 255.
   */
  public int defaultAmplification() { return 1; }


}