/*****************************************************************************
 $Id: StegoAlgoLSB.java,v 1.21 2002/04/14 03:42:33 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.Algo;

import java.awt.*;
import java.awt.image.*;
import javax.swing.*;

import trex.Filter.*;
import trex.*;

/**
 * This class implements the LSB Steganography Algorithm.
 * Data is split into bits; the picture's red, green and blue share each carry
 * one bit of data in each pixel.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.21 $
 */

public class StegoAlgoLSB extends StegoAlgoRGB {

  /**
   * LSBEncryptFilter is the encrypt filter for the LSB algorithm.
   */
  protected class LSBEncryptFilter extends TRexFilterRGB {

    boolean finished;
    /**
     * Constructs a new filter.
     * @param data Data to hide in the picture
     * @param w picture's width
     * @param h picture's height
     */
    public LSBEncryptFilter(String data, int w, int h) {
      // simply call the TRexFilterRGB constructor
      super(data, w, h);
      finished = false;
    }

    /**
     * hides 3 bits of data in exactely one pixel and returns the new pixel
     * @param x horizontal coordinate
     * @param y vertical coordinate
     * @param rgb input pixel color
     * @return New rgb value.
     */
    public int filterRGB(int x, int y, int rgb) {

      if (finished) return rgb;
      // extract a/r/g/b from rgb
      byte[] color = TRexUtil.intToBytes(rgb);
      int bit1 = 0, bit2 = 0, bit3 = 0;

      int pos = x + (y*w);  // current pixel in picture
      try {  // more data needed?
        bit1 = TRexUtil.getBit(data, (pos*3)+0);
        bit2 = TRexUtil.getBit(data, (pos*3)+1);
        bit3 = TRexUtil.getBit(data, (pos*3)+2);
      }

      catch (ArrayIndexOutOfBoundsException e) {
        finished = true;
      }

      finally { /* Even in case of overflow, store data */

        if (bit1 == 0) color[1] = (byte)(color[1] & ~1);
        else           color[1] = (byte)(color[1] | 1);

        if (bit2 == 0) color[2] = (byte)(color[2] & ~1);
        else           color[2] = (byte)(color[2] | 1);

        if (bit3 == 0) color[3] = (byte)(color[3] & ~1);
        else           color[3] = (byte)(color[3] | 1);

      return TRexUtil.bytesToInt(color);
      }
    }
  }

  /**
   * LSBDecryptFilter is the decrypt filter for the LSB algorithm.
   */
  protected class LSBDecryptFilter extends TRexFilterRGB {

    int datalen;
    boolean failure = false;
    /**
     * Extracts the data from the image.
     * The data is stored privately and can be requested via the getData()
     * method.
     * @param x horizontal coordinate
     * @param y vertical coordinate
     * @param rgb input pixel color
     * @see trex.Filter.TRexFilterRGB#getData
     * @return Arbitrary rgb value, currently original color
     */
    public int filterRGB(int x, int y, int rgb) {
      if (failure) return rgb;
      int pos = x + (y*w);

      if ((datalen < 0) && (pos > 11)) {  // after 11 pixels, data size is known
        datalen = TRexUtil.bytesToInt(TRexUtil.subArray(data, 0, 4));
        if (datalen < 0) failure = true;
      }

      if ((datalen > 0) && ((pos-1)*3 > (datalen+4)*8))
        // this will be more than enough...?
        return rgb;

      byte colors[] = TRexUtil.intToBytes(rgb); // convert rgb to color vector

      int bit1 = colors[1] & 1;  // get LSB from color shares
      int bit2 = colors[2] & 1;
      int bit3 = colors[3] & 1;

      try { // set bits
        TRexUtil.setBit(data, (pos*3)+0, bit1);
        TRexUtil.setBit(data, (pos*3)+1, bit2);
        TRexUtil.setBit(data, (pos*3)+2, bit3);
      }
      /*
       * enough data should be allocated; this is just for the sake of it... :->
       */
      catch (ArrayIndexOutOfBoundsException e) { failure = true; }

      // return arbitrary pixel
      return rgb;
    }

    /**
     * Constructs a new filter.
     * @param w picture's width
     * @param h picture's height
     */
    public LSBDecryptFilter(int w, int h) {
      super(null, w, h);
      data = new byte[w*h+4];  // this is more than enough. so what?
      for (int i = 0; i < data.length; i++) data[i] = 0; // fill data with 0
                // should be done automatically - we'll see.
      datalen = -1;
    }
  }

  /**
   * Return whether the config setup in the setup/passphrase dialog is valid.
   * There is no setup in LSB -> true.
   * @return true.
   */
  public boolean validConfig() {
    return true;
  }

  /**
   * Return whether this algorithm needs a pass phrase.
   * LSB doesn't have a pass phrase -> false.
   * @return false.
   */
  public boolean hasPassPhrase() {
    return false;
  }

  /**
   * Return whether this algorithm has a config dialog.
   * LSB doesn't have a config dialog -> false.
   * @return false.
   */
  public boolean hasConfigDialog() {
    return false;
  }

  /**
   * LSB does not have a config dialog - returns <code>null</code>.
   * @return null.
   */
  public java.awt.Component getConfigDialog() { return null; }

  /**
   * Return an info string about this algorithm.
   * Contains the algorithm's name
   * @return "Least significant bit algorithm"
   */
  public String getInfo() {
    return "Least significant bit algorithm";
  }

  /**
   * Return whether the passed picture is large enough to carry the data
   * @param img Image to hide data in
   * @param data data to hide
   * @return Whether the <code>img</code> is large enough to carry <code>data</code>.
   */
  public boolean pictureLargeEnough(ImageIcon img, String data) {
    int picsize = img.getIconHeight()*img.getIconWidth();
    int datasize = data.length()+4;

    // We need 1 pixel for 3 bits of data
    return (picsize*3 >= datasize*8);
  }

  /**
   * Return a decrypt filter for this algorithm. Actually called by "virtual"
   * method {@link trex.Algo.StegoAlgoRGB#getDecrypted}.
   * @param w Picture's width
   * @param h Picture's height
   * @return The {@link java.awt.image.ImageFilter} that does all the work.
   */
  protected TRexFilterRGB getDecryptFilter(int w, int h) {
    return new LSBDecryptFilter(w, h);
  }

  /**
   * Return an encrypt filter for this algorithm. Actually called by "virtual"
   * method {@link trex.Algo.StegoAlgoRGB#getDecrypted}.
   * @param data Data to hide while filtering
   * @param w Picture's width
   * @param h Picture's height
   * @return The {@link java.awt.image.ImageFilter} that does all the work.
   */
  protected TRexFilterRGB getEncryptFilter(String data, int w, int h) {
    return new LSBEncryptFilter(data, w, h);
  }

}