/*****************************************************************************
 $Id: CombinedPanel.java,v 1.10 2002/04/28 21:37:26 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.GUI;

import javax.swing.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import trex.*;
import javax.swing.event.*;

/**
 * <p>This class provides a panel for the combined view, i.e. the uncrypted,
 * the crypted and the differential picture on the same screen.
 * It's main constructor takes two images and creates the third one.</p>
 *
 * <p>The three images are visualized in a three-pane-view:</p>
 * <code>
 * +---+---+<br>
 * | U | C |<br>
 * +---+---+<br>
 * | D | &nbsp; |<br>
 * +---+---+<br>
 * </code>
 * <p>Where the "U" is the uncrypted, "C" the crypted and "D" the differential
 * image.</p>
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.10 $
 */

public class CombinedPanel extends JPanel {

  /** Parent frame for message boxes */
  Frame parent;
  /** The three images */
  ImageIcon img1, img2, imgDiff;
  /** Labels containing the ImageIcons */
  JLabel diffLabel, img1Label, img2Label;
  /** State variable to prevent event recursion */
  boolean scrollingActive = false;

  /** The three scroll panes to contain the ImageIcons (Labels) */
  JScrollPane jScrollPane1 = new JScrollPane();
  JScrollPane jScrollPane2 = new JScrollPane();
  JScrollPane jScrollPaneDiff = new JScrollPane();

  /** The Panel to contain the control widgets */
  JPanel jControlPanel = new JPanel();

  /** The JScrollPane's scrollbars in an array ([v/h][0-2]) */
  JScrollBar scrollBars[][] = new JScrollBar[2][3];

  /** Whole panel's GridBagLayout */
  GridBagLayout PanelGridBagLayout = new GridBagLayout();

  /** Color amplification; defaults to full amplification */
  int amplification = 255;
  /** Image zoom; defaults to no zoom */
  int zoom = 1;
  /** amplification Slider */
  JSlider jSliderAmp = new JSlider();
  /** zoom Slider */
  JSlider jSliderZoom = new JSlider();
  /** Text "Amplification" label */
  JLabel jAmpLabel = new JLabel();
  /** Text "Zoom" label */
  JLabel jZoomLabel = new JLabel();
  /** Control widget pane's GridBagLayout */
  GridBagLayout ControlPanelGridBagLayout = new GridBagLayout();

  /**
   * Converts an int to a byte, setting the byte to 255 when input is larger
   * than 255, or 0, if input is negative.
   * @param in input
   * @return the clamped byte
   */
  public static final byte clamp(int in) {
    if (in > 255)
      return (byte)255;
    else if (in < 0)
      return (byte)0;
    else
      return (byte)in;
  }

  /**
   * return a differential pic of img1 and img2. The resulting color is
   * amplified with the amplification parameter (0..255).
   * @param img1 First image
   * @param img2 Second image
   * @param parent parent window (for message boxes) (null results in no warnings)
   * @param amplification Color amplification of diff image
   */
  protected static ImageIcon createDiff(ImageIcon img1,
                                        ImageIcon img2,
                                        Frame parent,
                                        int amplification) {
    int w1 = img1.getIconWidth();
    int h1 = img1.getIconHeight();
    int w2 = img2.getIconWidth();
    int h2 = img2.getIconHeight();

    if ((h1 != h2) || (w1 != w2)) {
      /* Who knows who will abuse this class... :) */
      if (parent != null)
        JOptionPane.showMessageDialog(parent,
                        "Pictures have different size. Can't construct differential picture.",
                        "Error converting data",
                        JOptionPane.ERROR_MESSAGE);
      return null;
    }

    /* allocate memory for pixel arrays of img1, img2 and imgDiff */
    int[] pixels1 = new int[h1*w1];
    int[] pixels2 = new int[h1*w1];
    int[] pixelsDiff = new int[h1*w1];

    /* Get img1's and img2's pixels */
    PixelGrabber pg1 = new PixelGrabber(img1.getImage(), 0, 0, w1, h1, pixels1, 0, w1);
    PixelGrabber pg2 = new PixelGrabber(img2.getImage(), 0, 0, w2, h2, pixels2, 0, w2);

    try {
      pg1.grabPixels();
      pg2.grabPixels();
    }

    catch (Exception e) {
      if (parent != null)
        JOptionPane.showMessageDialog(parent,
                        "An unknown error occured transforming image data. Sorry.",
                        "Error converting data",
                        JOptionPane.ERROR_MESSAGE);
      else
        TRexUtil.printDebug("Error! Could not fetch data.");
      return null;
    }

    byte pix1[] = new byte[4];
    byte pix2[] = new byte[4];
    byte pixDiff[] = new byte[4];

    /* for all pixels in the images... */
    for (int i = 0; i < pixelsDiff.length; i++) {
      /**
       *
       * Yes
      if ((i % 1000) == 0)
        TRexUtil.printDebug("Got " + i + "pixels. Have 1/2: " + pixels1[i] + "/" + pixels2[i]);
      */
      pix1 = TRexUtil.intToBytes(pixels1[i]);
      pix2 = TRexUtil.intToBytes(pixels2[i]);

      pixDiff[0] = pix1[0];
      /* calculate their difference and amplify */
      pixDiff[1] = clamp(amplification*Math.abs(pix1[1]-pix2[1]));
      pixDiff[2] = clamp(amplification*Math.abs(pix1[2]-pix2[2]));
      pixDiff[3] = clamp(amplification*Math.abs(pix1[3]-pix2[3]));
      /* set new pixel color */
      pixelsDiff[i] = TRexUtil.bytesToInt(pixDiff);
    }

    /* create a new BufferedImage with the new pixel data */
    BufferedImage bi = new BufferedImage(w1, h1, BufferedImage.TYPE_INT_RGB);
    bi.setRGB(0, 0, w1, h1, pixelsDiff, 0, w1);

    /* return an ImageIcon with this picture */
    return new ImageIcon(bi);

  }

  /**
   * Constructor. Takes the two images and a parent window (for message boxes).
   * @param img1 First image
   * @param img2 Second image
   * @param parent Parent frame
   */
  public CombinedPanel(ImageIcon img1, ImageIcon img2, Frame parent) {
    this(img1, img2, parent, 255);
  }

  /**
   * Constructor.
   * @param img1 First image.
   * @param img2 Second image.
   * @param amplification Default amplification.
   * @param parent Parent frame
   */
  public CombinedPanel(ImageIcon img1, ImageIcon img2, Frame parent, int amplification) {
    this.amplification = amplification;
    /* store data */
    this.img1 = img1; this.img2 = img2; this.parent = parent;
    this.imgDiff = null;

    /* If images are unset, display a message in panel */
    if ((img1 == null) || (img2 == null))
      insertTextLabel("Panel for combined view (Image unset).");
    else {
      /* create diff img and initialize Panel */
      this.imgDiff = createDiff(img1, img2, parent, amplification);
      try {
        jbInit();
      }
      catch(Exception e) {
        e.printStackTrace();
      }
    }
  }

  /**
   * Helper function: add a Label to this Panel
   * @param text Text to display
   */
  public void insertTextLabel(String text) {
    JLabel textLabel = new JLabel(text);
    textLabel.setHorizontalAlignment(JLabel.CENTER);

    this.setLayout(new GridLayout(1,1));
    this.add(textLabel);
  }

  /**
   * Initialize window.
   * Mainly produced by JBuilder Designer.
   */
  private void jbInit() throws Exception {
    this.setLayout(PanelGridBagLayout);
    this.setToolTipText("");
    this.add(jScrollPane1, new GridBagConstraints(0, 0, 1, 1, 1.0, 1.0,
             GridBagConstraints.NORTHWEST,
             GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
    this.add(jScrollPane2, new GridBagConstraints(1, 0, 1, 1, 1.0, 1.0,
             GridBagConstraints.CENTER,
             GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
    this.add(jScrollPaneDiff, new GridBagConstraints(0, 1, 1, 1, 1.0, 1.0,
            GridBagConstraints.CENTER,
            GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
    this.add(jControlPanel,      new GridBagConstraints(1, 1, 1, 1, 1.0, 1.0,
             GridBagConstraints.CENTER,
             GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));

    jAmpLabel.setText("Amplification");
    jAmpLabel.setVerticalAlignment(SwingConstants.TOP);
    jZoomLabel.setText("Zoom");
    jZoomLabel.setVerticalAlignment(SwingConstants.TOP);
    jSliderAmp.setMajorTickSpacing(20);
    jSliderAmp.setSnapToTicks(false);
    jSliderAmp.setMinimum(1);
    jSliderAmp.setMaximum(255);
    jSliderAmp.setPaintTicks(true);
    jSliderAmp.setPreferredSize(new Dimension(100, 16));
    jSliderAmp.addChangeListener(new javax.swing.event.ChangeListener() {
      public void stateChanged(ChangeEvent e) {
        jSliderAmp_stateChanged(e);
      }
    });
    jSliderZoom.setMajorTickSpacing(2);
    jSliderZoom.setSnapToTicks(true);
    jSliderZoom.setMinimum(1);
    jSliderZoom.setMaximum(10);
    jSliderZoom.setPaintTicks(true);
    jSliderZoom.setPreferredSize(new Dimension(100, 16));
    jSliderZoom.addChangeListener(new javax.swing.event.ChangeListener() {
      public void stateChanged(ChangeEvent e) {
        jSliderZoom_stateChanged(e);
      }
    });

    jControlPanel.setLayout(ControlPanelGridBagLayout);
    jControlPanel.add(jAmpLabel,  new GridBagConstraints(0, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
    jControlPanel.add(jSliderAmp, new GridBagConstraints(1, 0, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));
    jControlPanel.add(jZoomLabel, new GridBagConstraints(0, 1, 1, 1, 0.0, 0.0
            ,GridBagConstraints.WEST, GridBagConstraints.NONE, new Insets(0, 0, 0, 0), 0, 0));
    jControlPanel.add(jSliderZoom, new GridBagConstraints(1, 1, 1, 1, 0.0, 0.0
            ,GridBagConstraints.CENTER, GridBagConstraints.BOTH, new Insets(0, 0, 0, 0), 0, 0));




    ////////// auto-built code ends here.
    /* Preset sliders */
    jSliderAmp.setValue(amplification);
    jSliderZoom.setValue(zoom);

    /* Get JScrollPane's JScrollBars */
    scrollBars[0][0] = jScrollPane1.getHorizontalScrollBar();
    scrollBars[1][0] = jScrollPane1.getVerticalScrollBar();
    scrollBars[0][1] = jScrollPane2.getHorizontalScrollBar();
    scrollBars[1][1] = jScrollPane2.getVerticalScrollBar();
    scrollBars[0][2] = jScrollPaneDiff.getHorizontalScrollBar();
    scrollBars[1][2] = jScrollPaneDiff.getVerticalScrollBar();

    /* Add this as listener to scrollbars */
    for (int i = 0; i < 2; i++)
      for (int j = 0; j < 3; j++) {
        if (scrollBars[i][j] != null)
          scrollBars[i][j].addAdjustmentListener(new AdjustmentListener()  {
            public void adjustmentValueChanged(AdjustmentEvent e) {
              scrollbarScrolled(e);
            }
          });
    }

    /* Create labels for ImageIcons and display them in their Panes */
    diffLabel = new JLabel(imgDiff);
    img1Label = new JLabel(img1);
    img2Label = new JLabel(img2);
    jScrollPane1.setViewportView(img1Label);
    jScrollPane2.setViewportView(img2Label);
    jScrollPaneDiff.setViewportView(diffLabel);

  }

  /**
   * Listener for scrollbar activity.
   */
  void scrollbarScrolled(AdjustmentEvent e) {
    /* Anti-event-recursion:
     * If scrollbar 1 was changed, it would change 2 and 3; they would
     * in turn try to change 1 */
    if (!scrollingActive) {
      scrollingActive = true;
      JScrollBar others[] = new JScrollBar[2];

      int hv, which; hv = 0; which = 0;

      /* Which scrollbar sent this event? */
      if (e.getSource() == scrollBars[0][0]) { hv = 0; which = 0; }
      if (e.getSource() == scrollBars[1][0]) { hv = 1; which = 0; }
      if (e.getSource() == scrollBars[0][1]) { hv = 0; which = 1; }
      if (e.getSource() == scrollBars[1][1]) { hv = 1; which = 1; }
      if (e.getSource() == scrollBars[0][2]) { hv = 0; which = 2; }
      if (e.getSource() == scrollBars[1][2]) { hv = 1; which = 2; }

      /* So the nominees are: */
      if (which == 0) { others[0] = scrollBars[hv][1]; others[1] = scrollBars[hv][2]; }
      if (which == 1) { others[0] = scrollBars[hv][0]; others[1] = scrollBars[hv][2]; }
      if (which == 2) { others[0] = scrollBars[hv][0]; others[1] = scrollBars[hv][1]; }

      /* Set the same value to the "others" */
      others[0].setValue(e.getValue());
      others[1].setValue(e.getValue());

      /* Scrolling is done. */
      scrollingActive = false;
    }
  }

  /**
   * Helper function.
   * Set new diff image with (potentially new) amplification.
   */
  private void updateDiffImg() {
    /* Null checks seem to be necessary. */
    if (diffLabel != null) {
      imgDiff = createDiff(img1, img2, parent, amplification);
      if (imgDiff != null) {
        updateImgsAsZoomed(false);
      }
    }
  }

  /**
   * Create zoomed images and set them in their Labels.
   * @param all If true, update img1, img2 and diff; else, only update diff
   */
  private void updateImgsAsZoomed(boolean all) {
    /* Images may be null during object initialization */
    if ((diffLabel == null) || (imgDiff == null)) return;
    if ((all) && ((img1Label == null) || (img1 == null) ||
                  (img2Label == null) || (img2 == null))) return;
    /* If zoom is not "original size" == 1... */
    if (zoom != 1) {
      /* calculate new image size */
      int newWidth = img1.getIconWidth()*zoom;
      if (newWidth == 0) return; /* Whoooops? Should not happen, but does :-( */

      ImageIcon zoomedDiff =
        new ImageIcon(imgDiff.getImage().getScaledInstance(newWidth,
                                                           -1,
                                                           Image.SCALE_REPLICATE));
      diffLabel.setIcon(zoomedDiff);

      if (all) {
        ImageIcon zoomedImg1 =
          new ImageIcon(img1.getImage().getScaledInstance(newWidth,
                                                          -1,
                                                          Image.SCALE_REPLICATE));

        ImageIcon zoomedImg2 =
          new ImageIcon(img2.getImage().getScaledInstance(newWidth,
                                                          -1,
                                                          Image.SCALE_REPLICATE));

        img1Label.setIcon(zoomedImg1);
        img2Label.setIcon(zoomedImg2);
      }
    } else {
      /* Zoom == 1 -> just set unzoomed images */
      diffLabel.setIcon(imgDiff);
      if (all) {
        img1Label.setIcon(img1);
        img2Label.setIcon(img2);
      }
    }
  }

  /**
   * Event handler for amplification slider.
   * @param e "Slider was moved" event.
   */
  void jSliderAmp_stateChanged(ChangeEvent e) {
    amplification = ((JSlider)e.getSource()).getValue();
    updateDiffImg();
  }

  /**
   * Event handler for zoom slider.
   * @param e "Slider was moved" event.
   */
  void jSliderZoom_stateChanged(ChangeEvent e) {
    zoom = ((JSlider)e.getSource()).getValue();
    updateImgsAsZoomed(true);
  }
}